package ru.bmstu.streamgenerator.producer;

import com.github.javafaker.Faker;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import ru.bmstu.streamgenerator.model.UserInfo;

@Component
public class StreamGenerator {

    private final KafkaTemplate<String, UserInfo> kafkaTemplate;
    private final String topicName;
    private final Faker faker = new Faker();

    @Value("${spring.kafka.template.message-per-second:1}")
    private volatile int messagesPerSecond;

    @Autowired
    public StreamGenerator(
            KafkaTemplate<String, UserInfo> kafkaTemplate,
            @Value("${spring.kafka.template.default-topic}") String topicName
    ) {
        this.kafkaTemplate = kafkaTemplate;
        this.topicName = topicName;
    }

    public void setMessagesPerSecond(int messagesPerSecond) {
        this.messagesPerSecond = messagesPerSecond;
    }

    @Scheduled(fixedRate = 1000)
    public void generateStream() {
        if (messagesPerSecond > 0) {
            final long startTime = System.currentTimeMillis();
            final double stepSize = 1000.0 / messagesPerSecond;

            double nextStepTime = startTime;
            int currentStep = 0;
            while (currentStep < messagesPerSecond) {
                if (System.currentTimeMillis() > nextStepTime) {
                    sendMessage();
                    nextStepTime += stepSize;
                    currentStep++;
                }
            }
        }
    }

    private void sendMessage() {
        final UserInfo userInfo = UserInfo.builder()
                .name(faker.name().firstName())
                .country(faker.country().name())
                .build();
        kafkaTemplate.send(topicName, userInfo);
    }

}
